<?php

namespace OM4\Zapier;

use Exception;
use OM4\Zapier\Admin\Dashboard;
use OM4\Zapier\Logger;
use OM4\Zapier\Plugin\CheckoutFieldEditor;
use OM4\Zapier\Plugin\Subscriptions;
use OM4\Zapier\SendQueue;
use OM4\Zapier\Trigger\TriggerFactory;
use WC_DateTime;

defined( 'ABSPATH' ) || exit;

/**
 * Class Plugin
 * This class is a singleton, and should be accessed via the WC_Zapier() function.
 *
 * @deprecated 2.0.0 Replaced by OM4\WooCommerceZapier\Plugin
 */
class Plugin {

	/**
	 * URL to the documentation for this legacy functionality.
	 */
	const DOCUMENTATION_URL = 'https://docs.om4.io/woocommerce-zapier/legacy/';

	/**
	 * Stores the one and only instance of this class
	 *
	 * @var self
	 */
	private static $instance;

	/**
	 * Dashboard instance.
	 *
	 * @var Dashboard
	 */
	private $admin;

	/**
	 * Full path to this plugin's directory (including trailing slash)
	 *
	 * @var string
	 */
	public static $plugin_dir;

	/**
	 * The list of WooCommerce order statuses.
	 *
	 * Generated by Plugin::get_order_statuses()
	 *
	 * @var string[]
	 */
	private static $order_statuses;

	/**
	 * SendQueue instance.
	 *
	 * @var SendQueue
	 */
	public static $queue;

	/**
	 * The main Plugin instance.
	 *
	 * @return self
	 */
	public static function instance() {

		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Initialise the plugin
	 * Constructor is private so nobody else can create an instance.
	 */
	private function __construct() {
		self::$plugin_dir = dirname( WC_ZAPIER_PLUGIN_FILE ) . '/';
	}

	/**
	 * Executed during the 'plugins_loaded' WordPress hook.
	 *
	 * - Checks that we're running the correct WooCommerce Version
	 * - Sets up various hooks
	 * - Load Supported Zapier Triggers
	 * - Loads the admin/dashboard interface if required
	 *
	 * @return void
	 */
	public function plugins_loaded() {

		self::$queue = SendQueue::instance();

		// User has a supported version of WooCommerce, so let's proceed.

		// WooCommerce init is priority 0, cancel_order is priority 10
		// Initialize in between those two so that cancelled orders are
		// accounted for.
		add_action( 'init', array( $this, 'init' ), 5 );

		// Check the WooCommerce Subscriptions plugin.
		if ( class_exists( 'WC_Subscriptions' ) ) {
			new Subscriptions();
		}
		// Check the WooCommerce Checkout Field Editor plugin.
		if ( function_exists( 'woocommerce_init_checkout_field_editor' ) ) {
			new CheckoutFieldEditor();
		}

		add_action( 'wc_zapier_resend_sample_data', array( '\\OM4\\Zapier\\Plugin', 'resend_sample_data' ) );
	}

	/**
	 * Obtain the list of WooCommerce order statuses.
	 *
	 * @return string[] Array of order status slugs (excluding the wc- prefix)
	 */
	public static function get_order_statuses() {
		if ( is_null( self::$order_statuses ) ) {
			self::$order_statuses = array();
			$statuses             = wc_get_order_statuses();
			foreach ( $statuses as $status => $status_label ) {
				// Use the order status without wc- internal prefix.
				$status                 = 'wc-' === substr( $status, 0, 3 ) ? substr( $status, 3 ) : $status;
				self::$order_statuses[] = $status;
			}
		}
		return self::$order_statuses;
	}

	/**
	 * Registers the custom post type for storing Legacy Zapier feeds.
	 * Executed during the 'init' WordPress hook.
	 *
	 * @return void
	 */
	public function init() {
		// The Custom Post Type that stores the zapier feeds.
		register_post_type(
			'wc_zapier_feed',
			array(
				'public'          => false,
				'show_ui'         => true,
				'show_in_menu'    => false,
				'supports'        => false,
				'labels'          => array(
					'name'               => __( 'Legacy Feeds', 'woocommerce-zapier' ),
					'singular_name'      => __( 'Legacy Feed', 'woocommerce-zapier' ),
					'add_new_item'       => __( 'Add New Legacy Feed', 'woocommerce-zapier' ),
					'edit_item'          => __( 'Edit Legacy Feed', 'woocommerce-zapier' ),
					'new_item'           => __( 'New Legacy Feed', 'woocommerce-zapier' ),
					'view_item'          => __( 'View Legacy Feed', 'woocommerce-zapier' ),
					'search_items'       => __( 'Search Legacy Feeds', 'woocommerce-zapier' ),
					'not_found'          => __( 'No Legacy Feeds found', 'woocommerce-zapier' ),
					'not_found_in_trash' => __( 'No Legacy Feeds found in Trash', 'woocommerce-zapier' ),
				),
				// Prevent *all* users from creating new Legacy Feeds, but allow existing Feeds to be edited and deleted by Shop Owners.
				'map_meta_cap'    => true,
				'capability_type' => 'shop_order',
				'capabilities'    => array(
					'create_posts' => 'do_not_allow',
				),
			)
		);

		TriggerFactory::load_triggers();

		if ( is_admin() ) {
			$this->admin = new Dashboard();
		}
	}

	/**
	 * Return a formatted price (excluding currency symbols) In the output, the
	 * number of decimal places matching WooCommerce's "General -> Currency
	 * options -> Number of decimals" setting
	 *
	 * @param int|float $price The unformatted price (without any thousands separators).
	 *
	 * @return string The formatted price.
	 */
	public static function format_price( $price ) {
		return wc_format_decimal( $price, wc_get_price_decimals() );
	}

	/**
	 * Format a WordPress date into a W3C formatted date in the store's local
	 * timezone (eg 2005-08-15T15:52:01+08:00)
	 *
	 * @param string|WC_DateTime $date Date to format to.
	 *
	 * @return string Date formatted like 2005-08-15T15:52:01+08:00.
	 */
	public static function format_date( $date ) {
		if ( ! is_a( $date, 'WC_DateTime' ) && empty( $date ) ) {
			return '';
		}

		if ( is_a( $date, 'WC_DateTime' ) ) {
			// A WC 2.7+ date - convert it to a date/time string in the site's local timezone.
			$date = gmdate( 'Y-m-d H:i:s', $date->getOffsetTimestamp() );
		}
		return date_i18n( DATE_W3C, strtotime( $date ) );
	}

	/**
	 * Prepare (decode) a string, in preparation for sending to Zapier
	 * Some order fields in WooCommerce are HTML entity encoded, and they need
	 * to be decoded before sending to Zapier. For example, "&amp;" should be
	 * sent as "&".
	 *
	 * @see: https://github.com/woocommerce/woocommerce/commit/07237d9a09849e107707943453b1fb245b8b4a2d
	 * @see: https://github.com/woocommerce/woocommerce/issues/10149
	 *
	 * @param string|array $value string|array The string/array that needs to be decoded.
	 *
	 * @return string|array The decoded string/array.
	 */
	public static function decode( $value ) {
		if ( is_array( $value ) ) {
			return json_encode( array_map( array( 'OM4\\Zapier\\Plugin', 'decode' ), $value ) );
		} elseif ( is_object( $value ) ) {
			// TODO: Handle scenarios where an object is passed (Issue #125).
			( new Logger() )->notice( 'OM4\Zapier\Plugin::decode() value is an object: %s', array( json_encode( $value ) ) );
			return json_encode( $value );
		} else {
			return html_entity_decode( $value, ENT_QUOTES, get_bloginfo( 'charset' ) );
		}
	}

	/**
	 * Asynchronously send sample data to all Feeds that use the specified trigger(s)
	 * This ensures that all existing Zaps include the latest data specification.
	 *
	 * @param strings[] $trigger_keys List of trigger keys.
	 *
	 * @return void
	 */
	public static function resend_sample_data_async( $trigger_keys = array() ) {
		wp_schedule_single_event( time(), 'wc_zapier_resend_sample_data', array( 'trigger_keys' => $trigger_keys ) );
	}

	/**
	 * Immediately re-send sample data to all currently active Zapier Feeds that use these triggers
	 * This ensures that Zapier has the latest checkout field definitions.
	 *
	 * @param string[] $trigger_keys Array of trigger keys.
	 *
	 * @return void
	 */
	public static function resend_sample_data( $trigger_keys = array() ) {
		if ( empty( $trigger_keys ) ) {
			$trigger_keys = TriggerFactory::get_trigger_keys();
		}
		foreach ( $trigger_keys as $trigger_key ) {
			try {
				$trigger = TriggerFactory::get_trigger_with_key( $trigger_key );
				$trigger->send_sample_data_to_active_feeds_using_this_trigger();
			} catch ( Exception $ex ) {
				// Invalid trigger key - silently ignore and continue.
				continue;
			}
		}
	}
}
