<?php

/**
 * Rest API - Rule Controller.
 * 
 * @since 9.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('FGF_Rule_Controller')) {

	/**
	 * Class
	 */
	class FGF_Rule_Controller extends WC_REST_Controller {

		/**
		 * Endpoint namespace.
		 *
		 * @var string
		 */
		protected $namespace = 'fgf';

		/**
		 * Route base.
		 *
		 * @var string
		 */
		protected $rest_base = 'rules';

		/**
		 * Post type.
		 *
		 * @var string
		 */
		protected $post_type = FGF_Register_Post_Types::RULES_POSTTYPE;

		/**
		 * Register the Rest API.
		 */
		public function register_routes() {
			register_rest_route($this->namespace, '/' . $this->rest_base, array(
				'methods' => 'GET',
				'callback' => array($this, 'get_items'),
				'permission_callback' => array($this, 'get_items_permissions_check'),
					)
			);

			register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)',
					array(
						'methods' => 'GET',
						'callback' => array($this, 'get_item'),
						'permission_callback' => array($this, 'get_item_permissions_check'),
					)
			);
		}

		/**
		 * Get object.
		 *
		 * @param int $id Object ID.
		 *
		 * @return Object
		 */
		protected function get_object( $id) {
			return fgf_get_rule($id);
		}

		/**
		 * Checks if a given request has access to get a specific item.
		 *
		 * @param WP_REST_Request $request Full details about the request.
		 * @return true|WP_Error True if the request has read access for the item, WP_Error object otherwise.
		 */
		public function get_item_permissions_check( $request) {
			$object = $this->get_object((int) $request['id']);

			if ($object && $object->exists() && !wc_rest_check_post_permissions($this->post_type, 'read', $object->get_id())) {
				return new WP_Error('woocommerce_rest_cannot_view', __('Sorry, you cannot view this resource.', 'woocommerce'), array('status' => rest_authorization_required_code()));
			}

			return true;
		}

		/**
		 * Checks if a given request has access to get a specific item.
		 *
		 * @param WP_REST_Request $request Full details about the request.
		 * @return true|WP_Error True if the request has read access for the item, WP_Error object otherwise.
		 */
		public function get_items_permissions_check( $request) {
			if (!wc_rest_check_post_permissions($this->post_type, 'read')) {
				return new WP_Error('woocommerce_rest_cannot_view', __('Sorry, you cannot list resources.', 'woocommerce'), array('status' => rest_authorization_required_code()));
			}

			return true;
		}

		/**
		 * Retrieves one item from the collection.
		 *         
		 * @param WP_REST_Request $request Full details about the request.
		 * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
		 */
		public function get_item( $request) {
			$rule_id = (int) $request['id'];
			$rule = $this->get_object($rule_id);
			if (!$rule->exists()) {
				return new WP_Error('fgf_rest_rules_invalid_id', __('Invalid rule ID', 'free-gifts-for-woocommerce'), array('status' => 404));
			}

			$response = $this->prepare_item_for_response($rule, $request);

			return rest_ensure_response($response);
		}

		/**
		 * Get all rules.
		 *
		 * @param WP_REST_Request $request Full details about the request.
		 * @return WP_Error|WP_REST_Response
		 */
		public function get_items( $request) {
			$args = array(
				'post_type' => $this->post_type,
				'post_status' => fgf_get_rule_statuses(),
				'posts_per_page' => '-1',
				'fields' => 'ids',
				'orderby' => 'menu_order',
				'order' => 'ASC',
			);

			$rule_ids = get_posts($args);

			$rules = array();
			foreach ($rule_ids as $rule_id) {
				$rule = $this->get_object($rule_id);
				if (!$rule->exists()) {
					continue;
				}

				$rules[] = $this->prepare_item_for_response($rule, $request);
			}

			return rest_ensure_response($rules);
		}

		/**
		 * Prepare a single customer output for response.
		 *
		 * @param  WP_User          $user_data User object.
		 * @param  WP_REST_Request  $request Request object.
		 * @return WP_REST_Response $response Response data.
		 */
		public function prepare_item_for_response( $rule, $request) {

			$response = array(
				'id' => $rule->get_id(),
				'name' => $rule->get_name(),
				'status' => $rule->get_status(),
				'description' => $rule->get_description(),
				'created_date' => $rule->get_created_date(),
				'modified_date' => $rule->get_modified_date(),
				'rule_type' => $rule->get_rule_type(),
				'gift_type' => $rule->get_gift_type(),
				'gift_products' => $rule->get_gift_products(),
				'gift_categories' => $rule->get_gift_categories(),
				'bogo_gift_type' => $rule->get_bogo_gift_type(),
				'buy_product_type' => $rule->get_buy_product_type(),
				'buy_product' => $rule->get_buy_product(),
				'buy_categories' => $rule->get_buy_categories(),
				'buy_category_type' => $rule->get_buy_category_type(),
				'get_product_type' => $rule->get_product_type(),
				'get_products' => $rule->get_products(),
				'get_categories' => $rule->get_categories(),
				'buy_quantity_type' => $rule->get_buy_quantity_type(),
				'buy_product_count' => $rule->get_buy_product_count(),
				'get_product_count' => $rule->get_product_count(),
				'bogo_gift_repeat' => $rule->get_bogo_gift_repeat(),
				'bogo_gift_repeat_mode' => $rule->get_bogo_gift_repeat_mode(),
				'bogo_gift_repeat_limit' => $rule->get_bogo_gift_repeat_limit(),
				'apply_coupon' => $rule->get_apply_coupon(),
				'coupon_gift_products' => $rule->get_coupon_gift_products(),
				'coupon_gift_products_qty' => $rule->get_coupon_gift_products_qty(),
				'rule_valid_from_date' => $rule->get_rule_valid_from_date(),
				'rule_valid_to_date' => $rule->get_rule_valid_to_date(),
				'rule_week_days_validation' => $rule->get_rule_week_days_validation(),
				'automatic_product_qty' => $rule->get_automatic_product_qty(),
				'rule_gifts_count_per_order' => $rule->get_rule_gifts_count_per_order(),
				'rule_usage_count' => $rule->get_rule_usage_count(),
				'rule_restriction_count' => $rule->get_rule_restriction_count(),
				'rule_allowed_user_type' => $rule->get_rule_allowed_user_type(),
				'rule_allowed_user_count' => $rule->get_rule_allowed_user_count(),
				'rule_allowed_user_usage_count' => $rule->get_rule_allowed_user_usage_count(),
				'condition_type' => $rule->get_condition_type(),
				'total_type' => $rule->get_total_type(),
				'cart_categories' => $rule->get_cart_categories(),
				'cart_subtotal_min_value' => $rule->get_cart_subtotal_minimum_value(),
				'cart_subtotal_max_value' => $rule->get_cart_subtotal_maximum_value(),
				'quantity_min_value' => $rule->get_quantity_minimum_value(),
				'quantity_max_value' => $rule->get_quantity_maximum_value(),
				'product_count_min_value' => $rule->get_product_count_min_value(),
				'product_count_max_value' => $rule->get_product_count_max_value(),
				'show_notice' => $rule->get_show_notice(),
				'notice' => $rule->get_notice(),
				'user_filter_type' => $rule->get_user_filter_type(),
				'include_users' => $rule->get_include_users(),
				'exclude_users' => $rule->get_exclude_users(),
				'include_user_roles' => $rule->get_include_user_roles(),
				'exclude_user_roles' => $rule->get_exclude_user_roles(),
				'product_filter_type' => $rule->get_product_filter_type(),
				'include_products' => $rule->get_include_products(),
				'include_product_count' => $rule->get_include_product_count(),
				'exclude_products' => $rule->get_exclude_products(),
				'applicable_products_type' => $rule->get_applicable_products_type(),
				'applicable_categories_type' => $rule->get_applicable_categories_type(),
				'include_categories' => $rule->get_include_categories(),
				'include_category_product_count' => $rule->get_include_category_product_count(),
				'exclude_categories' => $rule->get_exclude_categories()
			);

			/**
			 * This hook is used to alter the item response.
			 * 
			 * @since 9.0
			 */
			return apply_filters('fgf_prepared_rule_item_response', $response, $rule, $request);
		}

	}

}
	
