<?php

/**
 * Template functions.
 * */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}


if (!function_exists('fgf_get_template')) {

	/**
	 * Get the other templates from themes.
	 */
	function fgf_get_template( $template_name, $args = array()) {
		wc_get_template($template_name, $args, 'free-gifts-for-woocommerce/', FGF()->templates());
	}

}

if (!function_exists('fgf_get_template_html')) {

	/**
	 *  Like fgf_get_template, but returns the HTML instead of outputting.
	 *
	 *  @return string
	 */
	function fgf_get_template_html( $template_name, $args = array()) {
		ob_start();
		fgf_get_template($template_name, $args);
		return ob_get_clean();
	}

}

if (!function_exists('fgf_get_pagination_classes')) {

	/**
	 * Get the pagination classes.
	 *
	 *  @return array
	 */
	function fgf_get_pagination_classes( $page_no, $current_page) {
		$classes = array('fgf_pagination', 'fgf_pagination_' . $page_no);
		if ($current_page == $page_no) {
			$classes[] = 'current';
		}
		/**
		 * This hook is used to alter the pagination classes.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_pagination_classes', $classes, $page_no, $current_page);
	}

}

if (!function_exists('fgf_get_pagination_number')) {

	/**
	 * Get the pagination number.
	 *
	 *  @return string
	 */
	function fgf_get_pagination_number( $start, $page_count, $current_page) {
		$page_no = false;
		if ($current_page <= $page_count && $start <= $page_count) {
			$page_no = $start;
		} else if ($current_page > $page_count) {
			$overall_count = $current_page - $page_count + $start;
			if ($overall_count <= $current_page) {
				$page_no = $overall_count;
			}
		}
		/**
		 * This hook is used to alter the pagination number.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_pagination_number', $page_no, $start, $page_count, $current_page);
	}

}

if (!function_exists('fgf_get_gift_product_heading_label')) {

	/**
	 * Get the label for gift product heading.
	 *
	 * @return string.
	 * */
	function fgf_get_gift_product_heading_label() {
		/**
		 * This hook is used to alter the gift product heading label.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_gift_product_heading_label', get_option('fgf_settings_free_gift_heading_label'));
	}

}

if (!function_exists('fgf_get_gift_product_add_to_cart_button_label')) {

	/**
	 * Get the label for gift product add to cart button.
	 *
	 * @return string.
	 * */
	function fgf_get_gift_product_add_to_cart_button_label() {
		/**
		 * This hook is used to alter the gift product add to cart button label.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_gift_product_add_to_cart_button_label', get_option('fgf_settings_free_gift_add_to_cart_button_label'));
	}

}

if (!function_exists('fgf_get_gift_product_dropdown_default_value_label')) {

	/**
	 * Get the label for gift product dropdown default value.
	 *
	 * @return string.
	 * */
	function fgf_get_gift_product_dropdown_default_value_label() {
		/**
		 * This hook is used to alter the gift product dropdown default value label.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_gift_product_dropdown_default_value_label', get_option('fgf_settings_free_gift_dropdown_default_option_label', 'Please select a Gift'));
	}

}

if (!function_exists('fgf_get_dropdown_gift_product_name')) {

	/**
	 * Get the dropdown gift product name.
	 * 
	 * @return string.
	 * */
	function fgf_get_dropdown_gift_product_name( $product_id, $product = false) {
		if (!is_object($product)) {
			$product = wc_get_product($product_id);
		}
		/**
		 * This hook is used to alter the drop down gift product name.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_get_dropdown_gift_product_name', $product->get_name(), $product);
	}

}

if (!function_exists('fgf_show_dropdown_add_to_cart_button')) {

	/**
	 * Show the dropdown add to cart button.
	 * 
	 * @return bool.
	 * */
	function fgf_show_dropdown_add_to_cart_button() {
		/**
		 * This hook is used to validate the drop down add to cart button.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_show_dropdown_add_to_cart_button', '2' != get_option('fgf_settings_dropdown_add_to_cart_behaviour'));
	}

}

if (!function_exists('fgf_render_product_name')) {

	/**
	 * Display the gift product name in table.
	 *
	 * @return string
	 */
	function fgf_render_product_name( $product, $echo = true) {
		$product_name = $product->get_name();
		if ('2' == get_option('fgf_settings_gift_display_product_linkable', '1')) {
			$product_name = "<a href='" . get_permalink($product->get_id()) . "'>" . esc_html($product_name) . '</a>';
		}
		/**
		 * This hook is used to alter the gift product name.
		 * 
		 * @since 1.0
		 */
		$product_name = apply_filters('fgf_gift_product_name', $product_name, $product);
		if ($echo) {
			echo wp_kses_post($product_name);
		}

		return $product_name;
	}

}

if (!function_exists('fgf_get_gift_product_add_to_cart_classes')) {

	/**
	 * Get the gift product add to cart classes.
	 *
	 *  @return array
	 */
	function fgf_get_gift_product_add_to_cart_classes() {
		$classes = array('button', 'fgf-add-manual-gift-product');
		/**
		 * This hook is used to alter the gift product add to cart classes.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_gift_product_add_to_cart_classes', $classes);
	}

}

if (!function_exists('fgf_get_gift_product_add_to_cart_url')) {

	/**
	 * Get the gift product add to cart URL.
	 *
	 *  @return array
	 */
	function fgf_get_gift_product_add_to_cart_url( $gift_product, $permalink = false) {
		if (!$permalink) {
			$permalink = get_permalink();
		}

		if ('yes' == get_option('fgf_settings_enable_ajax_add_to_cart')) {
			$url = '#';
		} else {
			$args = array(
				'fgf_gift_product' => $gift_product['product_id'],
				'fgf_rule_id' => $gift_product['rule_id'],
			);

			if (isset($gift_product['buy_product_id'])) {
				$args['fgf_buy_product_id'] = $gift_product['buy_product_id'];
			}

			$url = esc_url(add_query_arg($args, $permalink));
		}

		/**
		 * This hook is used to alter the gift product add to cart URL.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_gift_product_add_to_cart_url', $url);
	}

}

if (!function_exists('fgf_get_gift_products_popup_notice')) {

	/**
	 * Get the gift products popup notice.
	 * 
	 * @since 9.8.0
	 * @return string
	 */
	function fgf_get_gift_products_popup_notice() {
		$popup_link = '<a href="#" class="fgf-popup-gift-products">' . get_option('fgf_settings_free_gift_popup_link_message') . '</a>';
		$find_shortcodes = array('[popup_link]', '[remaining_gift_count]');
		$replace_shortcodes = array($popup_link, FGF_Rule_Handler::get_remaining_gift_products_count());

		/**
		 * This hook is used to alter the gift products popup notice.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_gift_products_popup_notice', str_replace($find_shortcodes, $replace_shortcodes, get_option('fgf_settings_free_gift_popup_notice_message')));
	}

}

if (!function_exists('fgf_get_manual_gift_products_notice')) {

	/**
	 * Get the gift products notice.
	 * 
	 * @since 9.8.0
	 * @return string
	 */
	function fgf_get_manual_gift_products_notice() {

		/**
		 * This hook is used to alter the manual gift products notice.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_manual_gift_products_notice', str_replace('[remaining_gift_count]', FGF_Rule_Handler::get_remaining_gift_products_count(), get_option('fgf_settings_free_gift_notice_message')));
	}

}

if (!function_exists('fgf_get_progress_bar_heading_label')) {

	/**
	 * Get the label for progress bar heading.
	 *
	 * @since 9.8.0
	 * @return string.
	 * */
	function fgf_get_progress_bar_heading_label() {
		/**
		 * This hook is used to alter the progress bar heading label.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_progress_bar_heading_label', get_option('fgf_settings_progress_bar_heading_label'));
	}

}


if (!function_exists('fgf_get_progress_bar_maximum_gift_count_label')) {

	/**
	 * Get the label for progress bar maximum gift count.
	 *
	 * @since 9.8.0
	 * @return string.
	 * */
	function fgf_get_progress_bar_maximum_gift_count_label() {
		/**
		 * This hook is used to alter the progress bar maximum gift count label.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_progress_bar_maximum_gift_count_label', str_replace('[maximum_gift_count]', intval(FGF_Rule_Handler::get_total_gift_products_count()), get_option('fgf_settings_progress_bar_maximum_gift_count_label')));
	}

}

if (!function_exists('fgf_get_progress_bar_added_gift_count_label')) {

	/**
	 * Get the label for progress bar added gift count.
	 *
	 * @since 9.8.0
	 * @return string.
	 * */
	function fgf_get_progress_bar_added_gift_count_label() {
		/**
		 * This hook is used to alter the progress bar added gift count label.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_progress_bar_added_gift_count_label', str_replace('[added_gift_count]', intval(FGF_Rule_Handler::get_added_gift_products_count()), get_option('fgf_settings_progress_bar_added_gift_count_label')));
	}

}

if (!function_exists('fgf_get_progress_bar_remaining_gift_count_label')) {

	/**
	 * Get the label for progress bar remaining gift count.
	 *
	 * @since 9.8.0
	 * @return string.
	 * */
	function fgf_get_progress_bar_remaining_gift_count_label() {
		/**
		 * This hook is used to alter the progress bar remaining gift count label.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_progress_bar_remaining_gift_count_label', str_replace('[remaining_gift_count]', intval(FGF_Rule_Handler::get_remaining_gift_products_count()), get_option('fgf_settings_progress_bar_remaining_gift_count_label')));
	}

}
