<?php

/**
 * Common functions.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

include_once( 'fgf-layout-functions.php' );
include_once( 'fgf-post-functions.php' );
include_once( 'admin/fgf-admin-functions.php' );
include_once( 'fgf-template-functions.php' );
include_once( 'fgf-default-functions.php' );

if (!function_exists('fgf_check_is_array')) {

	/**
	 * Check if the resource is array.
	 *
	 * @return bool
	 */
	function fgf_check_is_array( $data) {
		return ( is_array($data) && !empty($data) );
	}

}

if (!function_exists('fgf_price')) {

	/**
	 *  Display Price based wc_price function
	 *
	 *  @return string
	 */
	function fgf_price( $price, $echo = true) {

		if ($echo) {
			echo wp_kses_post(wc_price($price));
		}

		return wc_price($price);
	}

}

if (!function_exists('fgf_render_product_image')) {

	/**
	 * Display the product image.
	 *
	 * @return mixed
	 */
	function fgf_render_product_image( $product, $size = 'woocommerce_thumbnail', $echo = true) {

		if ($echo) {
			echo wp_kses_post($product->get_image($size));
		}

		return $product->get_image();
	}

}

if (!function_exists('fgf_get_wc_cart_subtotal')) {

	/**
	 * Get the WC cart subtotal.
	 *
	 * @return string/float
	 */
	function fgf_get_wc_cart_subtotal() {
		if (!is_object(WC()->cart)) {
			return 0;
		}

		if (method_exists(WC()->cart, 'get_subtotal')) {
			$subtotal = ( 'incl' == get_option('woocommerce_tax_display_cart') ) ? WC()->cart->get_subtotal() + WC()->cart->get_subtotal_tax() : WC()->cart->get_subtotal();
		} else {
			$subtotal = ( 'incl' == get_option('woocommerce_tax_display_cart') ) ? WC()->cart->subtotal + WC()->cart->subtotal_tax : WC()->cart->subtotal;
		}

		return $subtotal;
	}

}

if (!function_exists('fgf_get_wc_cart_total')) {

	/**
	 * Get the WC cart total.
	 *
	 * @return string/float
	 */
	function fgf_get_wc_cart_total() {
		if (!is_object(WC()->cart)) {
			return 0;
		}

		if (version_compare(WC()->version, '3.2.0', '>=')) {
			$total = WC()->cart->get_total(true);
		} else {
			$total = WC()->cart->total;
		}

		return $total;
	}

}

if (!function_exists('fgf_get_free_gift_products_in_cart')) {

	/**
	 * Get the free gift products in the cart.
	 *
	 * @return array/int
	 */
	function fgf_get_free_gift_products_in_cart( $count = false, $automatic = false) {
		$free_gift_products = array();
		$free_gift_products_count = 0;

		if (is_object(WC()->cart)) {
			foreach (WC()->cart->get_cart() as $key => $value) {
				if (!isset($value['fgf_gift_product'])) {
					continue;
				}

				if ($automatic && 'automatic' == $value['fgf_gift_product']['mode']) {
					$value['fgf_gift_product']['quantity'] = $value['quantity'];
					$free_gift_products_count += $value['quantity'];

					if (isset($free_gift_products[$value['fgf_gift_product']['product_id']])) {

						$free_gift_products[$value['fgf_gift_product']['product_id']][$value['fgf_gift_product']['rule_id']] = $value['quantity'];
					} else {
						$free_gift_products[$value['fgf_gift_product']['product_id']] = array($value['fgf_gift_product']['rule_id'] => $value['quantity']);
					}
				} elseif (!$automatic && 'manual' == $value['fgf_gift_product']['mode']) {
					$value['fgf_gift_product']['quantity'] = $value['quantity'];
					$free_gift_products_count += $value['quantity'];

					if (isset($free_gift_products[$value['fgf_gift_product']['product_id']])) {

						$free_gift_products[$value['fgf_gift_product']['product_id']][$value['fgf_gift_product']['rule_id']] = $value['quantity'];
					} else {
						$free_gift_products[$value['fgf_gift_product']['product_id']] = array($value['fgf_gift_product']['rule_id'] => $value['quantity']);
					}
				}
			}
		}

		if ($count) {
			return $free_gift_products_count;
		}

		return $free_gift_products;
	}

}

if (!function_exists('fgf_get_bogo_products_count_in_cart')) {

	/**
	 * Get the BOGO products count in the cart.
	 *
	 * @return int
	 */
	function fgf_get_bogo_products_count_in_cart( $buy_product_id, $rule_id, $get_product_id = false, $mode = '3') {
		$quantity = 0;
		if (!is_object(WC()->cart)) {
			return $quantity;
		}

		$mode = ( '5' == $mode ) ? 'manual_bogo' : 'bogo';

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (!isset($value['fgf_gift_product']['mode']) || $mode != $value['fgf_gift_product']['mode']) {
				continue;
			}

			if ($rule_id != $value['fgf_gift_product']['rule_id']) {
				continue;
			}

			if ($buy_product_id != $value['fgf_gift_product']['buy_product_id']) {
				continue;
			}

			if ($get_product_id && $get_product_id != $value['fgf_gift_product']['product_id']) {
				continue;
			}

			$quantity += $value['quantity'];
		}

		return $quantity;
	}

}

if (!function_exists('fgf_get_coupon_gift_product_count_in_cart')) {

	/**
	 * Get the coupon gift product count in the cart.
	 *
	 * @return int
	 */
	function fgf_get_coupon_gift_product_count_in_cart( $product_id, $coupon_id, $rule_id) {
		$quantity = 0;
		if (!is_object(WC()->cart)) {
			return $quantity;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (!isset($value['fgf_gift_product']['mode']) || 'coupon' != $value['fgf_gift_product']['mode']) {
				continue;
			}

			if ($rule_id != $value['fgf_gift_product']['rule_id']) {
				continue;
			}

			if ($coupon_id != $value['fgf_gift_product']['coupon_id']) {
				continue;
			}

			if ($product_id != $value['fgf_gift_product']['product_id']) {
				continue;
			}

			$quantity += $value['quantity'];
		}

		return $quantity;
	}

}

if (!function_exists('fgf_get_free_gift_products_count_in_cart')) {

	/**
	 * Get the free gift products count in the cart.
	 *
	 * @return integer
	 */
	function fgf_get_free_gift_products_count_in_cart( $exclude_bogo = false) {
		$free_gift_products_count = 0;
		if (!is_object(WC()->cart)) {
			return $free_gift_products_count;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (!isset($value['fgf_gift_product'])) {
				continue;
			}

			if ($exclude_bogo && ( !isset($value['fgf_gift_product']['mode']) || 'bogo' == $value['fgf_gift_product']['mode'] )) {
				continue;
			}

			$value['fgf_gift_product']['quantity'] = $value['quantity'];
			$free_gift_products_count += $value['quantity'];
		}

		return $free_gift_products_count;
	}

}

if (!function_exists('fgf_get_rule_products_count_in_cart')) {

	/**
	 * Get the rule products count in Cart
	 *
	 * @return int
	 */
	function fgf_get_rule_products_count_in_cart( $rule_id) {
		$count = 0;
		if (!is_object(WC()->cart)) {
			return $count;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (!isset($value['fgf_gift_product'])) {
				continue;
			}

			if ($value['fgf_gift_product']['rule_id'] != $rule_id) {
				continue;
			}

			$count += $value['quantity'];
		}

		return $count;
	}

}

if (!function_exists('fgf_get_cart_item_count')) {

	/**
	 * Get the cart item count from the cart.
	 *
	 * @return int
	 */
	function fgf_get_cart_item_count( $exclude_gift = true) {
		$count = 0;
		if (!is_object(WC()->cart)) {
			return $count;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {
			if (isset($value['fgf_gift_product']) && $exclude_gift) {
				continue;
			}

			$count++;
		}

		return $count;
	}

}

if (!function_exists('fgf_get_wc_cart_category_subtotal')) {

	/**
	 * Get the category subtotal from the cart.
	 * 
	 * @param array $category_ids
	 * @param string $taxanomy
	 * @param boolean $consider_subcategories
	 * @param boolean $exclude_discount
	 * @return float
	 */
	function fgf_get_wc_cart_category_subtotal( $category_ids, $taxanomy = 'product_cat', $consider_subcategories = false, $exclude_discount = false) {
		$cart_total = 0;
		if (!fgf_check_is_array($category_ids)) {
			return $cart_total;
		}

		if (!is_object(WC()->cart)) {
			return $cart_total;
		}

		$overall_category_ids = $category_ids;
		if ($consider_subcategories) {
			foreach ($category_ids as $category_id) {
				$child_categories = get_categories(array('taxonomy' => $taxanomy, 'child_of' => $category_id, 'hide_empty' => true));
				$child_category_ids = array_column($child_categories, 'term_id');
				$overall_category_ids = array_filter(array_unique(array_merge($overall_category_ids, $child_category_ids)));
			}
		}

		$tax_display_cart = get_option('woocommerce_tax_display_cart');
		foreach (WC()->cart->get_cart() as $key => $value) {
			if (isset($value['fgf_gift_product'])) {
				continue;
			}

			$product_categories = get_the_terms($value['product_id'], $taxanomy);
			if (!fgf_check_is_array($product_categories)) {
				continue;
			}

			foreach ($product_categories as $product_category) {
				if (in_array($product_category->term_id, $overall_category_ids)) {
					if ($exclude_discount) {
						$cart_total += ( 'incl' == $tax_display_cart ) ? $value['line_total'] + $value['line_tax'] : $value['line_total'];
					} else {
						$cart_total += ( 'incl' == $tax_display_cart ) ? $value['line_subtotal'] + $value['line_subtotal_tax'] : $value['line_subtotal'];
					}

					break;
				}
			}
		}

		return $cart_total;
	}

}

if (!function_exists('fgf_get_product_count_in_cart')) {

	/**
	 * Get the product count in the cart.
	 *
	 * @return int
	 */
	function fgf_get_product_count_in_cart( $product_id) {
		$product_count = 0;
		if (!is_object(WC()->cart)) {
			return $product_count;
		}

		foreach (WC()->cart->get_cart() as $key => $value) {

			$cart_product_id = !empty($value['variation_id']) ? $value['variation_id'] : $value['product_id'];

			if ($cart_product_id != $product_id) {
				continue;
			}

			$product_count += $value['quantity'];
		}

		return $product_count;
	}

}

if (!function_exists('fgf_get_address_metas')) {

	/**
	 * Get the user address meta(s).
	 *
	 * @return array
	 */
	function fgf_get_address_metas( $flag) {

		$address_metas = array(
			'first_name',
			'last_name',
			'company',
			'address_1',
			'address_2',
			'city',
			'country',
			'postcode',
			'state',
		);

		return 'billing' == $flag ? array_merge($address_metas, array('email', 'phone')) : $address_metas;
	}

}

if (!function_exists('fgf_get_address')) {

	/**
	 * Get the user address.
	 *
	 * @return array
	 */
	function fgf_get_address( $user_id, $flag) {
		$billing_metas = fgf_get_address_metas($flag);

		foreach ($billing_metas as $each_meta) {
			$billing_address[$each_meta] = get_user_meta($user_id, $flag . '_' . $each_meta, true);
		}

		return $billing_address;
	}

}

if (!function_exists('fgf_get_free_gifts_per_page_column_count')) {

	/**
	 * Get the free gifts per page column count.
	 *
	 * @return int
	 */
	function fgf_get_free_gifts_per_page_column_count() {
		// To avoid pagination if the table pagination is disabled.
		$display_table_pagination = get_option('fgf_settings_gift_display_table_pagination');
		if ('2' == $display_table_pagination) {
			return 10000;
		}

		$per_page = get_option('fgf_settings_free_gift_per_page_column_count', 4);

		if (!$per_page) {
			return 4;
		}

		return $per_page;
	}

}

if (!function_exists('fgf_get_carousel_options')) {

	/**
	 * Get the carousel options.
	 *
	 * @return array
	 */
	function fgf_get_carousel_options() {

		// Declare values.
		$nav = ( 'yes' == get_option('fgf_settings_carousel_navigation') ) ? true : false;
		$auto_play = ( 'yes' == get_option('fgf_settings_carousel_auto_play') ) ? true : false;
		$pagination = ( 'yes' == get_option('fgf_settings_carousel_pagination') ) ? true : false;
		$nav_prev_text = get_option('fgf_settings_carousel_navigation_prevoius_text');
		$nav_next_text = get_option('fgf_settings_carousel_navigation_next_text');
		$desktop_count = get_option('fgf_settings_carousel_gift_per_page', 3);
		$tablet_count = get_option('fgf_settings_carousel_gift_per_page_tablet', 2);
		$mobile_count = get_option('fgf_settings_carousel_gift_per_page_mobile', 1);
		$item_margin = get_option('fgf_settings_carousel_item_margin');
		$item_per_slide = get_option('fgf_settings_carousel_item_per_slide');
		$slide_speed = get_option('fgf_settings_carousel_slide_speed');

		$nav_prev_text = ( empty($nav_prev_text) ) ? '<' : $nav_prev_text;
		$nav_next_text = ( empty($nav_next_text) ) ? '<' : $nav_next_text;
		$desktop_count = ( empty($desktop_count) ) ? '3' : $desktop_count;
		$tablet_count = ( empty($tablet_count) ) ? '2' : $tablet_count;
		$mobile_count = ( empty($mobile_count) ) ? '1' : $mobile_count;
		$item_margin = ( empty($item_margin) ) ? '10' : $item_margin;
		$item_per_slide = ( empty($item_per_slide) ) ? '1' : $item_per_slide;
		$slide_speed = ( empty($slide_speed) ) ? '5000' : $slide_speed;

		return array(
			'desktop_count' => $desktop_count,
			'tablet_count' => $tablet_count,
			'mobile_count' => $mobile_count,
			'item_margin' => $item_margin,
			'nav' => json_encode($nav),
			'nav_prev_text' => $nav_prev_text,
			'nav_next_text' => $nav_next_text,
			'pagination' => json_encode($pagination),
			'item_per_slide' => $item_per_slide,
			'slide_speed' => $slide_speed,
			'auto_play' => json_encode($auto_play),
		);
	}

}

if (!function_exists('fgf_get_rule_translated_string')) {

	/**
	 * Get the rule translated string.
	 *
	 * @return mixed
	 */
	function fgf_get_rule_translated_string( $option_name, $value, $language = null) {
		/**
		 * This hook is used to alter the rule translated string.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_rule_translate_string', $value, $option_name, $language);
	}

}

if (!function_exists('fgf_get_product')) {

	/**
	 * Get the product object by product id.
	 *
	 * @return object/bool
	 */
	function fgf_get_product( $product_id) {
		/**
		 * This hook is used to validate the product.
		 * 
		 * @since 1.0
		 */
		if (!apply_filters('fgf_is_valid_product', true, $product_id)) {
			return false;
		}
		/**
		 * This hook is used to alter the product.
		 * 
		 * @since 1.0
		 */
		return apply_filters('fgf_get_product', wc_get_product($product_id), $product_id);
	}

}

if (!function_exists('fgf_rule_product_exists')) {

	/**
	 * Check the rule product exists?
	 *
	 * @return bool
	 */
	function fgf_rule_product_exists( $rule, $product_id, $buy_product_id = false) {
		$exists = false;
		if (!is_object($rule)) {
			return $exists;
		}

		switch ($rule->get_rule_mode()) {
			case 'manual':
				// Manual.
				if (FGF_Rule_Handler::rule_product_exists($rule->get_id(), $product_id)) {
					$exists = true;
				}
				break;

			case 'manual_bogo':
				// Manual Bogo.
				if (FGF_Rule_Handler::get_bogo_rule_product_qty($rule->get_id(), $product_id, $buy_product_id)) {
					$exists = true;
				}
				break;
		}

		return $exists;
	}

}

if (!function_exists('fgf_get_term_ids')) {

	/**
	 * Get the term ids.
	 *
	 * @return array
	 */
	function fgf_get_term_ids( $object, $taxonomy = 'product_cat') {
		if (is_numeric($object)) {
			$object_id = $object;
		} else {
			$object_id = $object->get_id();
		}

		$terms = get_the_terms($object_id, $taxonomy);
		if (false === $terms || is_wp_error($terms)) {
			return array();
		}

		return wp_list_pluck($terms, 'term_id');
	}

}

if (!function_exists('fgf_add_html_inline_style')) {

	/**
	 * Add the custom CSS to HTML elements.
	 *
	 * @since 8.4
	 * @return Mixed
	 */
	function fgf_add_html_inline_style( $content, $css, $full_content = false) {
		if (!$css || !$content) {
			return $content;
		}

		// Return the content with style css when DOMDocument class not exists.
		if (!class_exists('DOMDocument')) {
			return '<style type="text/css">' . $css . '</style>' . $content;
		}

		if (class_exists('\Pelago\Emogrifier\CssInliner')) {
			// To create a instance with original HTML.
			$css_inliner_class = 'Pelago\Emogrifier\CssInliner';
			$domDocument = $css_inliner_class::fromHtml($content)->inlineCss($css)->getDomDocument();
			// Removing the elements with display:none style declaration from the content.
			$html_pruner_class = 'Pelago\Emogrifier\HtmlProcessor\HtmlPruner';
			$html_pruner_class::fromDomDocument($domDocument)->removeElementsWithDisplayNone();
			// Converts a few style attributes values to visual HTML attributes.
			$attribute_converter_class = 'Pelago\Emogrifier\HtmlProcessor\CssToAttributeConverter';
			$visual_html = $attribute_converter_class::fromDomDocument($domDocument)->convertCssToVisualAttributes();

			$content = ( $full_content ) ? $visual_html->render() : $visual_html->renderBodyContent();
		} elseif (class_exists('\Pelago\Emogrifier')) {
			$emogrifier_class = 'Pelago\Emogrifier';
			$emogrifier = new Emogrifier($content, $css);
			$content = ( $full_content ) ? $emogrifier->emogrify() : $emogrifier->emogrifyBodyContent();
		} elseif (version_compare(WC_VERSION, '4.0', '<')) {
			$emogrifier_class = 'Emogrifier';
			if (!class_exists($emogrifier_class)) {
				include_once dirname(WC_PLUGIN_FILE) . '/includes/libraries/class-emogrifier.php';
			}

			$emogrifier = new Emogrifier($content, $css);
			$content = ( $full_content ) ? $emogrifier->emogrify() : $emogrifier->emogrifyBodyContent();
		}

		return $content;
	}

}

if (!function_exists('fgf_get_rule_valid_gift_products')) {

	/**
	 * Get the rule valid gift products.
	 *
	 * @since 9.6.0
	 * @return array
	 */
	function fgf_get_rule_valid_gift_products() {
		/**
		 * This hook is used to alter the rule valid gift products.
		 * 
		 * @since 9.6.0
		 */
		return apply_filters('fgf_rule_valid_gift_products', FGF_Rule_Handler::get_valid_gift_products());
	}

}

if (!function_exists('fgf_get_manual_gift_products_progress_bar_width')) {

	/**
	 * Get the manual gift products progress bar width.
	 * 
	 * @since 9.8.0
	 * @return float
	 */
	function fgf_get_manual_gift_products_progress_bar_width() {
		$width = 0;
		if (FGF_Rule_Handler::get_added_gift_products_count()) {
			$width = ( intval(FGF_Rule_Handler::get_added_gift_products_count()) / intval(FGF_Rule_Handler::get_total_gift_products_count()) * 100 );
		}

		/**
		 * This hook is used to alter the manual gift products progress bar width.
		 * 
		 * @since 9.8.0
		 */
		return apply_filters('fgf_manual_gift_products_progress_bar_width', $width);
	}

}
