<?php

/**
 * Compatibility - WooCommerce Brands.
 * 
 * @since 8.6
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('FGF_WC_Brands_Compatibility')) {

	/**
	 * Class.
	 */
	class FGF_WC_Brands_Compatibility extends FGF_Compatibility {

		/**
		 * Taxonomy.
		 * 
		 * @var string
		 */
		const TAXONOMY = 'product_brand';

		/**
		 * Class Constructor.
		 */
		public function __construct() {
			$this->id = 'wc_brands';

			parent::__construct();
		}

		/**
		 * Is plugin enabled?.
		 * 
		 *  @return bool
		 * */
		public function is_plugin_enabled() {
			return class_exists('WC_Brands');
		}

		/**
		 * Admin Action.
		 */
		public function admin_action() {
			// Add the brands fields in the rule.
			add_action('fgf_after_rule_criteria_total_type_settings', array($this, 'add_brands_fields'));
			add_action('fgf_after_rule_product_filters_settings', array($this, 'add_product_filter_brands_fields'));
			// Add the brand in the rule criteria total type option.
			add_filter('fgf_rule_criteria_total_type_options', array($this, 'add_custom_criteria_total_type_option'), 20, 1);
			// Add the brand in the rule product filter option.
			add_filter('fgf_rule_product_filter_options', array($this, 'add_custom_product_filter_option'), 20, 1);
			// Validate and prepare the brand rule fields.
			add_filter('fgf_prepare_rule_post_data', array($this, 'prepare_rule_post_data'), 20, 1);
		}

		/**
		 * Frontend Action.
		 */
		public function frontend_action() {
			// May be alter the cart criteria total based on selection.
			add_action('fgf_rule_cart_criteria_total', array($this, 'maybe_alter_cart_criteria_total'), 100, 2);

			// May be validate the rule if the brands are not valid.
			add_action('fgf_rule_product_category_filter', array($this, 'maybe_validate_rule_brands_filter'), 20, 2);
		}

		/**
		 * Add the brands fields in the rule.
		 * 
		 * @param array $rule_data
		 */
		public function add_brands_fields( $rule_data) {
			$brands = $this->get_wc_brands();
			$brand_ids = isset($rule_data['id']) ? array_filter((array) get_post_meta($rule_data['id'], 'fgf_cart_brands', true)) : array();

			include_once( FGF_ABSPATH . 'inc/compatibility/views/html-rule-criteria-brand-total-fields.php' );
		}

		/**
		 * Add the product filter brands fields in the rule.
		 * 
		 * @since 9.4.0
		 * @param array $rule_data
		 */
		public function add_product_filter_brands_fields( $rule_data) {
			$brands = $this->get_wc_brands();

			$include_brands = isset($rule_data['id']) ? array_filter((array) get_post_meta($rule_data['id'], 'fgf_include_brands', true)) : array();
			$exclude_brands = isset($rule_data['id']) ? array_filter((array) get_post_meta($rule_data['id'], 'fgf_exclude_brands', true)) : array();
			$applicable_brands_type = isset($rule_data['id']) ? get_post_meta($rule_data['id'], 'fgf_applicable_brands_type', true) : '1';
			$brand_product_count = isset($rule_data['id']) ? get_post_meta($rule_data['id'], 'fgf_brand_product_count', true) : '1';

			include_once( FGF_ABSPATH . 'inc/compatibility/views/html-rule-product-brand-filters.php' );
		}

		/**
		 * Add the brand in the rule criteria total type option.
		 * 
		 * @param array $options
		 * @return array
		 */
		public function add_custom_criteria_total_type_option( $options) {
			$options['brands'] = __('Brands Total', 'free-gifts-for-woocommerce');

			return $options;
		}

		/**
		 * Add the brand in the rule product filter option.
		 * 
		 * @since 9.4.0
		 * @param array $options
		 * @return array
		 */
		public function add_custom_product_filter_option( $options) {
			$options['include_brands'] = __('Include Brands', 'free-gifts-for-woocommerce');
			$options['exclude_brands'] = __('Exclude Brands', 'free-gifts-for-woocommerce');

			return $options;
		}

		/**
		 * Validate and prepare the brand rule fields.
		 * 
		 * @param array $rule_post_data
		 * @return array
		 */
		public function prepare_rule_post_data( $rule_post_data) {
			$rule_post_data['fgf_cart_brands'] = isset($rule_post_data['fgf_cart_brands']) ? $rule_post_data['fgf_cart_brands'] : array();

			// Validate the brand selection.
			if ('brands' === $rule_post_data['fgf_total_type'] && empty($rule_post_data['fgf_cart_brands'])) {
				throw new Exception(__('Please select atleast one brand', 'free-gifts-for-woocommerce'));
			}

			$rule_post_data['fgf_include_brands'] = isset($rule_post_data['fgf_include_brands']) ? $rule_post_data['fgf_include_brands'] : array();
			$rule_post_data['fgf_exclude_brands'] = isset($rule_post_data['fgf_exclude_brands']) ? $rule_post_data['fgf_exclude_brands'] : array();

			if ('include_brands' === $rule_post_data['fgf_product_filter_type'] && empty($rule_post_data['fgf_include_brands'])) {
				throw new Exception(__('Please select atleast one Brand', 'free-gifts-for-woocommerce'));
			}

			return $rule_post_data;
		}

		/**
		 * May be alter the cart criteria total based on selection.
		 * 
		 * @param float $total
		 * @param object $rule
		 * @return float
		 */
		public function maybe_alter_cart_criteria_total( $total, $rule) {
			// return if the total type is not brands.
			if ('brands' !== $rule->get_total_type()) {
				return $total;
			}

			$brand_ids = array_filter((array) get_post_meta($rule->get_id(), 'fgf_cart_brands', true));

			return fgf_get_wc_cart_category_subtotal($brand_ids, self::TAXONOMY, $rule->get_consider_cart_subcategories_total(), $rule->is_exclude_category_subtotal_discount_amount());
		}

		/**
		 * May be validate the rule if the brands are not valid.
		 * 
		 * @since 9.4.0
		 * @param boolean $bool
		 * @param object $rule
		 * @return boolean
		 */
		public function maybe_validate_rule_brands_filter( $bool, $rule) {
			$brand_ids = array();
			$brand_product_count = 0;
			$applicable_brand_type = get_post_meta($rule->get_id(), 'fgf_applicable_brands_type', true);
			$included_brands = array_filter((array) get_post_meta($rule->get_id(), 'fgf_include_brands', true));

			foreach (WC()->cart->get_cart() as $cart_content) {
				if (isset($cart_content['fgf_gift_product'])) {
					continue;
				}

				switch ($rule->get_product_filter_type()) {
					case 'include_brands':
						$product_brands = get_the_terms($cart_content['product_id'], self::TAXONOMY);
						if (fgf_check_is_array($product_brands)) {
							foreach ($product_brands as $product_brand) {
								$current_brand_id = $product_brand->term_id;
								if ('1' === $applicable_brand_type && in_array($product_brand->term_id, $included_brands)) {
									return true;
								} elseif (in_array($product_brand->term_id, $included_brands)) {
									break;
								}
							}

							if (in_array($current_brand_id, $included_brands)) {
								$brand_product_count += $cart_content['quantity'];
							}

							$brand_ids[] = $current_brand_id;
						}
						break;

					case 'exclude_brands':
						$bool = true;
						$product_brands = get_the_terms($cart_content['product_id'], self::TAXONOMY);
						if (fgf_check_is_array($product_brands)) {
							$excluded_brands = array_filter((array) get_post_meta($rule->get_id(), 'fgf_exclude_brands', true));
							foreach ($product_brands as $product_brand) {
								if (in_array($product_brand->term_id, $excluded_brands)) {
									$bool = false;
									break;
								}
							}
						}

						break;
				}
			}

			if ('include_brands' === $rule->get_product_filter_type()) {
				if ('4' == $applicable_brand_type) {
					$bool = ( $brand_product_count >= floatval(get_post_meta($rule->get_id(), 'fgf_brand_product_count', true)) );
				} else {
					$bool = FGF_Rule_Handler::validate_applicable_product_category($applicable_brand_type, $included_brands, $brand_ids);
				}
			}

			return $bool;
		}

		/**
		 * Get the WC brands.
		 *
		 * @return array
		 */
		private function get_wc_brands() {
			$fgf_brands = array();
			$wc_brands = get_terms(self::TAXONOMY);

			if (!fgf_check_is_array($wc_brands)) {
				return $fgf_brands;
			}

			foreach ($wc_brands as $category) {
				$fgf_brands[$category->term_id] = $category->name;
			}

			return $fgf_brands;
		}

	}

}
