<?php

/**
 * Rules List Table.
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!class_exists('WP_List_Table')) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

if (!class_exists('FGF_Rules_List_Table')) {

	/**
	 * Class.
	 * */
	class FGF_Rules_List_Table extends WP_List_Table {

		/**
		 * Per page count.
		 * 
		 * @var int
		 * */
		private $perpage = 10;

		/**
		 * Database.
		 * 
		 * @var object
		 * */
		private $database;

		/**
		 * Offset.
		 * 
		 * @var int
		 * */
		private $offset;

		/**
		 * Order BY.
		 * 
		 * @var string
		 * */
		private $orderby = 'ORDER BY menu_order ASC,ID ASC';

		/**
		 * Post type.
		 * 
		 * @var string
		 * */
		private $post_type = FGF_Register_Post_Types::RULES_POSTTYPE;

		/**
		 * List Slug.
		 * 
		 * @var string
		 * */
		private $list_slug = 'fgf_rules';

		/**
		 * Base URL.
		 * 
		 * @var string
		 * */
		private $base_url;

		/**
		 * Current URL.
		 * 
		 * @var string
		 * */
		private $current_url;

		/**
		 * Constructor.
		 */
		public function __construct() {

			global $wpdb;
			$this->database = &$wpdb;

			// Prepare the required data.
			$this->base_url = fgf_get_rule_page_url();

			parent::__construct(
					array(
						'singular' => 'fgf_rule',
						'plural' => 'fgf_rules',
						'ajax' => false,
					)
			);
		}

		/**
		 * Prepares the list of items for displaying.
		 * */
		public function prepare_items() {
			// Prepare the current url.
			$this->current_url = add_query_arg(array('paged' => absint($this->get_pagenum())), $this->base_url);

			// Prepare the bulk actions.
			$this->process_bulk_action();

			// Prepare the perpage.
			$this->perpage = $this->get_items_per_page('fgf_rules_per_page');

			// Prepare the offset.
			$this->offset = $this->perpage * ( absint($this->get_pagenum()) - 1 );

			// Prepare the header columns.
			$this->_column_headers = array($this->get_columns(), $this->get_hidden_columns(), $this->get_sortable_columns());

			// Prepare the query clauses.
			$join = $this->get_query_join();
			$where = $this->get_query_where();
			$limit = $this->get_query_limit();
			$offset = $this->get_query_offset();
			$orderby = $this->get_query_orderby();

			// Prepare the all items.
			$count_items = $this->database->get_var('SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " AS p $where $orderby");

			// Prepare the current page items.
			$prepare_query = $this->database->prepare('SELECT DISTINCT ID FROM ' . $this->database->posts . " AS p $join $where $orderby LIMIT %d,%d", $offset, $limit);

			$items = $this->database->get_results($prepare_query, ARRAY_A);

			// Prepare the item object.
			$this->prepare_item_object($items);

			// Prepare the pagination arguments.
			$this->set_pagination_args(
					array(
						'total_items' => $count_items,
						'per_page' => $this->perpage,
					)
			);
		}

		/**
		 * Render the table.
		 * */
		public function render() {
			if (isset($_REQUEST['s']) && strlen(wc_clean(wp_unslash($_REQUEST['s'])))) { // @codingStandardsIgnoreLine.
				/* translators: %s: search keywords */
				echo wp_kses_post(sprintf('<span class="subtitle">' . __('Search results for &#8220;%s&#8221;', 'free-gifts-for-woocommerce') . '</span>', wc_clean(wp_unslash($_REQUEST['s']))));
			}

			// Output the table.
			$this->prepare_items();
			$this->views();
			$this->search_box(__('Search Rule', 'free-gifts-for-woocommerce'), 'fgf-rules');
			$this->display();
		}

		/**
		 * Get a list of columns.
		 * 
		 * @return array
		 * */
		public function get_columns() {
			$columns = array(
				'cb' => '<input type="checkbox" />', // Render a checkbox instead of text
				'rule_name' => __('Rule Name', 'free-gifts-for-woocommerce'),
				'description' => __('Description', 'free-gifts-for-woocommerce'),
				'status' => __('Status', 'free-gifts-for-woocommerce'),
				'validity' => __('Validity', 'free-gifts-for-woocommerce'),
				'type' => __('Type', 'free-gifts-for-woocommerce'),
				'product_category' => __('Product(s) / Categories', 'free-gifts-for-woocommerce'),
				'created_date' => __('Created Date', 'free-gifts-for-woocommerce'),
				'modified_date' => __('Last Modified Date', 'free-gifts-for-woocommerce'),
				'actions' => __('Actions', 'free-gifts-for-woocommerce'),
			);

			if (!isset($_REQUEST['post_status']) && !isset($_REQUEST['s'])) {
				$columns['sort'] = '<img src="' . esc_url(FGF_PLUGIN_URL . '/assets/images/drag-icon.png') . '" title="' . __('Sort', 'free-gifts-for-woocommerce') . '"></img>';
			}
			/**
			 * This hook is used to alter the rules columns.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_get_columns', $columns);
		}

		/**
		 * Get a list of hidden columns.
		 * 
		 * @return array
		 * */
		public function get_hidden_columns() {
			/**
			 * This hook is used to alter the rules hidden columns.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_hidden_columns', array());
		}

		/**
		 * Get a list of sortable columns.
		 * 
		 * @return void
		 * */
		public function get_sortable_columns() {
			/**
			 * This hook is used to alter the rules sortable columns.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_sortable_columns', array(
				'rule_name' => array('rule_name', false),
				'status' => array('status', false),
				'created_date' => array('created', false),
				'modified_date' => array('modified', false)
			));
		}

		/**
		 * Message to be displayed when there are no items.
		 */
		public function no_items() {
			esc_html_e('No rule to show.', 'free-gifts-for-woocommerce');
		}

		/**
		 * Get a list of bulk actions.
		 * 
		 * @return array
		 * */
		protected function get_bulk_actions() {
			$action = array();

			$action['active'] = __('Activate', 'free-gifts-for-woocommerce');
			$action['inactive'] = __('Deactivate', 'free-gifts-for-woocommerce');
			$action['delete'] = __('Delete', 'free-gifts-for-woocommerce');
			/**
			 * This hook is used to alter the rules bulk actions.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_bulk_actions', $action);
		}

		/**
		 * Display the list of views available on this table.
		 * 
		 * @return array
		 * */
		public function get_views() {
			$args = array();
			$status_link = array();
			/**
			 * This hook is used to alter the rules views.
			 * 
			 * @since 1.0
			 */
			$status_link_array = apply_filters($this->list_slug . '_get_views', array_filter(array_merge(array('all' => __('All', 'free-gifts-for-woocommerce')), fgf_get_rule_statuses_options())));

			foreach ($status_link_array as $status_name => $status_label) {
				$status_count = $this->get_total_item_for_status($status_name);

				if (!$status_count) {
					continue;
				}

				$args['status'] = $status_name;

				$label = $status_label . ' (' . $status_count . ')';

				$class = array(strtolower($status_name));
				if (isset($_GET['status']) && ( sanitize_title($_GET['status']) == $status_name )) { // @codingStandardsIgnoreLine.
					$class[] = 'current';
				}

				if (!isset($_GET['status']) && 'all' == $status_name) { // @codingStandardsIgnoreLine.
					$class[] = 'current';
				}

				$status_link[$status_name] = $this->get_edit_link($args, $label, implode(' ', $class));
			}

			return $status_link;
		}

		/**
		 * Get a edit link.
		 * 
		 * @rerurn string
		 * */
		private function get_edit_link( $args, $label, $class = '') {
			$url = add_query_arg($args, $this->base_url);
			$class_html = '';
			if (!empty($class)) {
				$class_html = sprintf(
						' class="%s"', esc_attr($class)
				);
			}

			return sprintf(
					'<a href="%s"%s>%s</a>', esc_url($url), $class_html, $label
			);
		}

		/**
		 * Get the total item by status.
		 * 
		 * @return int
		 * */
		private function get_total_item_for_status( $status = '') {
			// Get the current status item ids.
			$prepare_query = $this->database->prepare('SELECT COUNT(DISTINCT ID) FROM ' . $this->database->posts . " WHERE post_type=%s and post_status IN('" . $this->format_status($status) . "')", $this->post_type);

			return $this->database->get_var($prepare_query);
		}

		/**
		 * Format the status.
		 * 
		 * @return string
		 * */
		private function format_status( $status) {
			if ('all' == $status) {
				$statuses = fgf_get_rule_statuses();
				$status = implode("', '", $statuses);
			}

			return $status;
		}

		/**
		 * Bulk action functionality
		 * */
		public function process_bulk_action() {
			$ids = isset($_REQUEST['id']) ? wc_clean(wp_unslash(( $_REQUEST['id'] ))) : array(); // @codingStandardsIgnoreLine.
			$ids = !is_array($ids) ? explode(',', $ids) : $ids;

			if (!fgf_check_is_array($ids)) {
				return;
			}

			if (!current_user_can('edit_posts')) {
				wp_die('<p class="error">' . esc_html__('Sorry, you are not allowed to edit this item.', 'free-gifts-for-woocommerce') . '</p>');
			}

			$action = $this->current_action();

			foreach ($ids as $id) {
				if ('delete' === $action) {
					wp_delete_post($id, true);
				} elseif ('active' === $action) {
					fgf_update_rule($id, array(), array('post_status' => 'fgf_active'));
				} elseif ('inactive' === $action) {
					fgf_update_rule($id, array(), array('post_status' => 'fgf_inactive'));
				}
			}

			wp_safe_redirect($this->current_url);
			exit();
		}

		/**
		 * Prepare the CB column data.
		 * 
		 * @return string
		 * */
		protected function column_cb( $item) {
			return sprintf(
					'<input type="checkbox" name="id[]" value="%s" />', $item->get_id()
			);
		}

		/**
		 * Prepare a each column data.
		 * 
		 * @return mixed
		 * */
		protected function column_default( $item, $column_name) {

			switch ($column_name) {

				case 'rule_name':
					return '<a href="' . esc_url(
									add_query_arg(
											array(
												'action' => 'edit',
												'id' => $item->get_id(),
											), $this->base_url
									)
							) . '">' . esc_html($item->get_name()) . '</a>';
					break;

				case 'description':
					return $item->get_description();

					break;

				case 'status':
					return fgf_display_status($item->get_status());

					break;

				case 'type':
					return fgf_get_rule_type_name($item->get_rule_type());

					break;

				case 'validity':
					$from = !empty($item->get_rule_valid_from_date()) ? $item->get_rule_valid_from_date() : '-';
					$to = !empty($item->get_rule_valid_to_date()) ? $item->get_rule_valid_to_date() : '-';

					if ('-' === $from && '-' === $to) {
						return __('Unlimited', 'free-gifts-for-woocommerce');
					} elseif ('-' === $to) {
						$to = __('Unlimited', 'free-gifts-for-woocommerce');
					}

					return __('From', 'free-gifts-for-woocommerce') . '&nbsp:&nbsp&nbsp' . $from . '<br />' . __('To', 'free-gifts-for-woocommerce') . '&nbsp&nbsp&nbsp&nbsp&nbsp&nbsp&nbsp:&nbsp&nbsp' . $to;
					break;

				case 'product_category':
					return $this->render_product_category($item);
					break;

				case 'created_date':
					return $item->get_formatted_created_date();
					break;

				case 'modified_date':
					return $item->get_formatted_modified_date();
					break;

				case 'actions':
					$actions = array();
					$status_action = ( $item->get_status() == 'fgf_inactive' ) ? 'active' : 'inactive';

					$actions['edit'] = fgf_display_action('edit', $item->get_id(), $this->current_url, true);
					$actions[$status_action] = fgf_display_action($status_action, $item->get_id(), $this->current_url);
					$actions['delete'] = fgf_display_action('delete', $item->get_id(), $this->current_url);

					end($actions);

					$last_key = key($actions);
					$views = '';
					foreach ($actions as $key => $action) {
						$views .= $action;

						if ($last_key == $key) {
							break;
						}

						$views .= ' | ';
					}

					return $views;

					break;

				case 'sort':
					return '<div class = "fgf_post_sort_handle">'
							. '<img src = "' . esc_url(FGF_PLUGIN_URL . '/assets/images/drag-icon.png') . '" title="' . __('Sort', 'free-gifts-for-woocommerce') . '"></img>'
							. '<input type = "hidden" class = "fgf_rules_sortable" value = "' . $item->get_id() . '" />'
							. '</div>';
					break;
			}
		}

		/**
		 * Render the product category details.
		 * 
		 * @return string
		 */
		private function render_product_category( $item) {
			if ('4' == $item->get_rule_type()) {

				$coupon_label = __('Coupon Code to Apply', 'free-gifts-for-woocommerce');

				$coupons_link = $this->get_coupons_link($item->get_apply_coupon());
				$coupon_details = '<b><u>' . $coupon_label . '</b></u><br />' . $coupons_link;
				$products = $this->get_products_link($item->get_coupon_gift_products());
				$products_etails = '<br /><b><u>' . __('Gift Product(s)', 'free-gifts-for-woocommerce') . '</u></b> <br />' . $products;

				return $coupon_details . $products_etails;
			} elseif ('3' == $item->get_rule_type() || '5' == $item->get_rule_type()) {

				$buy_products_label = __('Buy Product(s)', 'free-gifts-for-woocommerce');
				if ('2' == $item->get_buy_product_type()) {
					$buy_products_link = $this->get_categories_link($item->get_buy_categories());
					$buy_products_link = __('Product(s) of', 'free-gifts-for-woocommerce') . ' ' . $buy_products_link;
				} else {
					$buy_products_link = $this->get_products_link($item->get_buy_product());
				}

				$bogo_products = '<b><u>' . $buy_products_label . '</u></b><br />' . $buy_products_link;
				$get_products_label = __('Get Product(s)', 'free-gifts-for-woocommerce');

				if ('5' == $item->get_rule_type() && '2' == $item->get_product_type()) {
					$get_products_link = $this->get_categories_link($item->get_categories());
					$get_products_link = __('Product(s) of', 'free-gifts-for-woocommerce') . ' ' . $get_products_link;
				} else if ('5' == $item->get_rule_type() || '2' == $item->get_bogo_gift_type()) {
					$get_products_link = $this->get_products_link($item->get_products());
				} else {
					$get_products_link = $buy_products_link;
				}

				$bogo_products .= '<br ><b><u>' . $get_products_label . '</u></b><br />' . $get_products_link;

				return $bogo_products;
			} elseif ('2' == $item->get_gift_type() && '2' != $item->get_rule_type()) {

				$categories_link = $this->get_categories_link($item->get_gift_categories());

				return '<b><u>' . __('Categories', 'free-gifts-for-woocommerce') . '</u></b><br />' . $categories_link;
			} else {
				$products_link = $this->get_products_link($item->get_gift_products());

				return '<b><u>' . __('Product(s)', 'free-gifts-for-woocommerce') . '</u></b><br />' . $products_link;
			}
		}

		/**
		 * Products Link.
		 * 
		 * @return string
		 * */
		private function get_products_link( $product_ids) {
			$products_link = '';

			foreach ($product_ids as $product_id) {
				$product = wc_get_product($product_id);

				//Return if the product does not exist.
				if (!$product) {
					continue;
				}

				$products_link .= '<a href="' . esc_url(
								add_query_arg(
										array(
											'post' => $product_id,
											'action' => 'edit',
										), admin_url('post.php')
								)
						) . '" >' . $product->get_name() . '</a> , ';
			}

			return rtrim($products_link, ' , ');
		}

		/**
		 * Categories Link.
		 * 
		 * @return string
		 * */
		private function get_categories_link( $categories_ids) {
			$categories_link = '';

			foreach ($categories_ids as $category_id) {
				$category = get_term_by('id', $category_id, 'product_cat');
				if (!is_object($category)) {
					continue;
				}

				$categories_link .= '<a href="' . esc_url(
								add_query_arg(
										array(
											'product_cat' => $category->slug,
											'post_type' => 'product',
										), admin_url('edit.php')
								)
						) . '" >' . $category->name . '</a> , ';
			}

			return rtrim($categories_link, ' , ');
		}

		/**
		 * Coupon Link.
		 * 
		 * @return string
		 * */
		private function get_coupons_link( $coupon_ids) {
			$coupons_link = '';

			foreach ($coupon_ids as $coupon_id) {
				$the_coupon = get_post($coupon_id);

				//Return if the coupon code does not exist.
				if (!$the_coupon) {
					continue;
				}

				$coupons_link .= '<a href="' . esc_url(
								add_query_arg(
										array(
											'post' => $coupon_id,
											'action' => 'edit',
										), admin_url('post.php')
								)
						) . '" >' . $the_coupon->post_title . '</a> , ';
			}

			return rtrim($coupons_link, ' , ');
		}

		/**
		 * Prepare the item Object.
		 * 
		 * @return void
		 * */
		private function prepare_item_object( $items) {
			$prepare_items = array();
			if (fgf_check_is_array($items)) {
				foreach ($items as $item) {
					$prepare_items[] = fgf_get_rule($item['ID']);
				}
			}

			$this->items = $prepare_items;
		}

		/**
		 * Get the query join clauses.
		 * 
		 * @return string
		 * */
		private function get_query_join() {
			$join = '';
			if (empty($_REQUEST['orderby'])) { // @codingStandardsIgnoreLine.
				return $join;
			}

			$join = ' INNER JOIN ' . $this->database->postmeta . ' AS pm ON ( pm.post_id = p.ID )';
			/**
			 * This hook is used to alter the rules join query fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_join', $join);
		}

		/**
		 * Get the query where clauses.
		 * 
		 * @return string
		 * */
		private function get_query_where() {
			$current_status = 'all';
			if (isset($_GET['status']) && ( sanitize_title($_GET['status']) != 'all' )) {
				$current_status = sanitize_title($_GET['status']);
			}

			$where = " where post_type='" . $this->post_type . "' and post_status IN('" . $this->format_status($current_status) . "')";

			// Search.
			$where = $this->custom_search($where);
			/**
			 * This hook is used to alter the rules where query fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_where', $where);
		}

		/**
		 * Get the query limit clauses.
		 * 
		 * @return string
		 * */
		private function get_query_limit() {
			/**
			 * This hook is used to alter the rules limit query fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_limit', $this->perpage);
		}

		/**
		 * Get the query offset clauses.
		 * 
		 * @return string
		 * */
		private function get_query_offset() {
			/**
			 * This hook is used to alter the rules offset query fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_offset', $this->offset);
		}

		/**
		 * Get the query order by clauses.
		 * 
		 * @return string
		 * */
		private function get_query_orderby() {

			$order = 'DESC';
			if (!empty($_REQUEST['order']) && is_string($_REQUEST['order'])) { // @codingStandardsIgnoreLine.
				if ('ASC' === strtoupper(wc_clean(wp_unslash($_REQUEST['order'])))) { // @codingStandardsIgnoreLine.
					$order = 'ASC';
				}
			}

			// Order By.
			if (isset($_REQUEST['orderby'])) {
				switch (wc_clean(wp_unslash($_REQUEST['orderby']))) { // @codingStandardsIgnoreLine.
					case 'rule_name':
						$this->orderby = ' ORDER BY p.post_title ' . $order;
						break;
					case 'status':
						$this->orderby = ' ORDER BY p.post_status ' . $order;
						break;
					case 'created':
						$this->orderby = ' ORDER BY p.post_date ' . $order;
						break;
					case 'modified':
						$this->orderby = ' ORDER BY p.post_modified ' . $order;
						break;
				}
			}
			/**
			 * This hook is used to alter the rules order by query fields.
			 * 
			 * @since 1.0
			 */
			return apply_filters($this->list_slug . '_query_orderby', $this->orderby);
		}

		/**
		 * Custom Search.
		 * 
		 * @retrun string
		 * */
		public function custom_search( $where) {

			if (!isset($_REQUEST['s'])) { // @codingStandardsIgnoreLine.
				return $where;
			}

			$post_ids = array();
			$terms = explode(' , ', wc_clean(wp_unslash($_REQUEST['s']))); // @codingStandardsIgnoreLine.

			foreach ($terms as $term) {
				$term = $this->database->esc_like(( $term ));
				$post_query = new FGF_Query($this->database->prefix . 'posts', 'p');
				$post_query->select('DISTINCT `p`.ID')
						->leftJoin($this->database->prefix . 'postmeta', 'pm', '`p`.`ID` = `pm`.`post_id`')
						->where('`p`.post_type', $this->post_type)
						->whereIn('`p`.post_status', fgf_get_rule_statuses())
						->whereLike('`p`.post_title', '%' . $term . '%');

				$post_ids = $post_query->fetchCol('ID');
			}

			$post_ids = fgf_check_is_array($post_ids) ? $post_ids : array(0);
			$where .= ' AND (id IN (' . implode(' , ', $post_ids) . '))';

			return $where;
		}

	}

}
